/* ------------------------------------------------------------------------ */
/* A2FCBMP.C (C) Copyright Bill Buckels 2012-2014                           */
/* All Rights Reserved.                                                     */
/*                                                                          */
/* Licence Agreement                                                        */
/* -----------------                                                        */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* If you don't agree, remove this source code from your computer now.      */
/*                                                                          */
/* Written by   : Bill Buckels                                              */
/* Email:         bbuckels@mts.net                                          */
/*                                                                          */
/* Purpose      : This utility will allow you to convert from 				*/
/*                Apple II Double Hi-Res 140 x 192 x 16 color images to:	*/
/*																			*/
/*		280 x 192 x 24 Bit Windows .BMP Files - Default						*/
/*		140 x 192 x 24 Bit Windows .BMP Files - Option "140"				*/
/*																			*/
/*      Option "P" (p0-p16) Color Mapping is in one of 13 optional palettes.*/
/*                The palette numbers are the same as Bmp2DHR:              */
/* 						  	                                                */
/*                          0 - Kegs32 (RGB)								*/
/*                          1 - CiderPress (RGB)							*/
/*                          2 - Old AppleWin (NTSC ?)						*/
/*                          3 - New AppleWin (NTSC ?)						*/
/*                          4 - Wikipedia (NTSC)							*/
/*                          5 -	tohgr (NTSC)								*/
/* 						  	palette 6 is a user palette file                */
/* 						  	palettes 7-11 are IBM-PC legacy palettes        */
/* 						  	palette 12 is Super Convert RGB palette         */
/* 						  	palette 13 is Jace NTSC palette                 */
/* 						  	palette 14 is Cybernesto's VBMP NTSC palette    */
/* 						  	palette 15 is merged RGB-NTSC colors (p5+p12)/2 */
/* 						  	palette 16 is tohgr's old NTSC colors           */
/*																			*/
/*                This utility will also allow you to convert from 			*/
/*                Apple II Double Hi-Res 560 x 192 monochrome images to:	*/
/*																			*/
/*		560 x 384 monochrome Windows .BMP Files - Option "384"				*/
/*		560 x 192 monochrome Windows .BMP Files - Option "192"				*/
/*																			*/
/*                This utility also converts from an AppleWin "Mono"        */
/*                screen capture 256 color BMP to any of the Above.         */
/*																			*/
/*                During conversion of the AppleWin 256 color BMP it also   */
/*                generates Apple II Double Hi-Res images and renames the 	*/
/*                AppleWin BMP.                             				*/
/*                                                                          */
/* Revision     : 3.0 Added AppleWin Support and odds and ends.             */
/*                4.0 Added Different Palettes 								*/
/*                    Added Long FileNames for 32 bit compilation			*/
/*                    Added optional Ciderpress tags for Long Filenames		*/
/* ------------------------------------------------------------------------ */
/* Originally Written in Large Model 16 bit Microsoft C (MSC) Version 8.00c */
/*                    Cleaned-up c++ code comments for old Turbo C 2.0      */
/*                    16-bit MS-DOS version now compiles under both         */
/*                    MSC and Turbo C                                       */
/* ------------------------------------------------------------------------ */
/*                    Added Conditional Compilation for MinGW and gcc       */
/*                    Added Conditional Compilation etc. for 32 bit support */
/*                    32-bit version now compiles under Visual Studio Win32 */
/*                    and MinGW Win32 and gcc with support for long names.  */
/*                    Should now run everywhere (Windows, Linux, OSX)       */
/* ------------------------------------------------------------------------ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

/* ------------------------------------------------------------------------ */
/* Declarations, Vars. etc.                                                 */
/* ------------------------------------------------------------------------ */

/* options flags */

int longnames = 1, bmp = 0, bm2 = 0, a2fc = 0, auxbin = 0, tohgr=0, mono = 0, doublepixel = 1, applesoft = 0,
    palnumber = 5, doublegrey = 0;
char fullname[256], shortname[256], outname[256], mainfile[256],auxfile[256],a2fcfile[256];

char *palname[] = {
	"CiderPress RGB",
	"Kegs32 RGB",
	"Old AppleWin NTSC",
	"New AppleWin NTSC",
	"Wikipedia NTSC",
	"tohgr NTSC DHGR",
	"Imported",
	"Legacy Canvas",
	"Legacy Win16",
	"Legacy Win32",
	"Legacy VGA BIOS",
	"Legacy VGA PCX",
	"Super Convert RGB",
	"Jace NTSC",
	"Cybernesto-Munafo NTSC",
	"Pseudo Palette",
	"tohgr NTSC HGR"};


typedef unsigned char uchar;
typedef unsigned short ushort;
typedef unsigned long ulong;
typedef short sshort;

/* Bitmap Header structures */
#ifdef MINGW
typedef struct __attribute__((__packed__)) tagRGBQUAD
#else
typedef struct tagRGBQUAD
#endif
{
    unsigned char    rgbBlue;
    unsigned char    rgbGreen;
    unsigned char    rgbRed;
    unsigned char    rgbReserved;
} RGBQUAD;


/* Bitmap Header structures */
#ifdef MINGW
typedef struct __attribute__((__packed__)) tagBITMAPINFOHEADER
#else
typedef struct tagBITMAPINFOHEADER
#endif
{
    ulong   biSize;
    ulong   biWidth;
    ulong   biHeight;
    ushort  biPlanes;
    ushort  biBitCount;
    ulong   biCompression;
    ulong   biSizeImage;
    ulong   biXPelsPerMeter;
    ulong   biYPelsPerMeter;
    ulong   biClrUsed;
    ulong   biClrImportant;
} BITMAPINFOHEADER;

#ifdef MINGW
typedef struct __attribute__((__packed__)) tagBITMAPFILEHEADER
#else
typedef struct tagBITMAPFILEHEADER
#endif
{
    uchar   bfType[2];
    ulong   bfSize;
    ushort  bfReserved1;
    ushort  bfReserved2;
    ulong   bfOffBits;
} BITMAPFILEHEADER;

/* constant for the biCompression field */
#define BI_RGB      0L

/* static structures for processing AppleWin bmp input file */
BITMAPFILEHEADER BitMapFileHeader;
RGBQUAD          sbmp[256]; /* super vga - applewin */
BITMAPINFOHEADER bmi;

#define ASCIIZ	0
#define CRETURN 13
#define LFEED	10

uchar *szTextTitle =
    "A2FCBMP(C) Version 4.0 Copyright Bill Buckels 2012-2014\n"
    "All Rights Reserved.";

#define SUCCESS  0
#define VALID    SUCCESS
#define FAILURE  -1
#define INVALID  FAILURE

#define NUM_RGB_COLORS     3
#define NUM_VGA_COLORS    16

/* 280 x 192 - the default output format */
uchar BMP_header[] ={
0x42, 0x4D, 0x36, 0x76, 0x02, 0x00, 0x00, 0x00,
0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x28, 0x00,
0x00, 0x00, 0x18, 0x01, 0x00, 0x00, 0xC0, 0x00,
0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x76, 0x02, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

/* 140 x 192 - verbatim output format */
uchar BMP140_header[] ={
0x42, 0x4D, 0x36, 0x3B, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x28, 0x00,
0x00, 0x00, 0x8C, 0x00, 0x00, 0x00, 0xC0, 0x00,
0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3B, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

/* monochrome output routines - ported from dmono */
/* headers for monochrome BMP files */

/* 560 x 384 - the default output format */
uchar mono384[62] ={
0x42, 0x4D, 0x3E, 0x6C, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x28, 0x00,
0x00, 0x00, 0x30, 0x02, 0x00, 0x00, 0x80, 0x01,
0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x6C, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00};

/* 560 x 192 - verbatim output format */
uchar mono192[62] ={
0x42, 0x4D, 0x3E, 0x36, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x28, 0x00,
0x00, 0x00, 0x30, 0x02, 0x00, 0x00, 0xC0, 0x00,
0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x00};

uchar msk[]={0x80,0x40,0x20,0x10,0x8,0x4,0x2,0x1};
uchar bmpscanline[72]; /* only the first 70 bytes are used */

uchar buf560[560];

/* built-in palette options */
/* based on DHGR, DLGR, LGR colors */
/* LGR color order */
uchar kegs32colors[16][3] = {
	0,0,0,		 /* black    */
	221,0,51,	 /* red      */
	0,0,153,	 /* dk blue  */
	221,0,221,	 /* purple   */
	0,119,0,	 /* dk green */
	85,85,85,	 /* gray     */
	34,34,255,	 /* med blue */
	102,170,255, /* lt blue  */
	136,85,34,	 /* brown    */
	255,102,0,	 /* orange   */
	170,170,170, /* grey     */
	255,153,136, /* pink     */
	0,221,0,	 /* lt green */
	255,255,0,	 /* yellow   */
	0,255,153,	 /* aqua     */
	255,255,255};/* white    */

uchar ciderpresscolors[16][3] = {
	0,0,0,		 /* black    */
	221,0,51,	 /* red      */
	0,0,153,	 /* dk blue  */
	221,34,221,	 /* purple   */
	0,119,34,	 /* dk green */
	85,85,85,	 /* gray     */
	34,34,255,	 /* med blue */
	102,170,255, /* lt blue  */
	136,85,0,	 /* brown    */
	255,102,0,	 /* orange   */
	170,170,170, /* grey     */
	255,153,136, /* pink     */
	17,221,0,	 /* lt green */
	255,255,0,	 /* yellow   */
	68,255,153,	 /* aqua     */
	255,255,255};/* white    */

uchar awinoldcolors[16][3] = {
	0,0,0,       /* black    */
	208,0,48,	 /* red      */
	0,0,128,	 /* dk blue  */
	255,0,255,	 /* purple   */
	0,128,0,	 /* dk green */
	128,128,128, /* gray     */
	0,0,255,	 /* med blue */
	96,160,255,	 /* lt blue  */
	128,80,0,	 /* brown    */
    255,128,0,	 /* orange   */
    192,192,192, /* grey     */
    255,144,128, /* pink     */
    0,255,0,	 /* lt green */
    255,255,0,	 /* yellow   */
    64,255,144,	 /* aqua     */
    255,255,255};/* white    */

uchar awinnewcolors[16][3] = {
	0,0,0,       /* black    */
	157,9,102,   /* red      */
	42,42,229,   /* dk blue  */
	199,52,255,  /* purple   */
	0,118,26,    /* dk green */
	128,128,128, /* gray     */
	13,161,255,  /* med blue */
	170,170,255, /* lt blue  */
	85,85,0,     /* brown    */
    242,94,0,    /* orange   */
    192,192,192, /* grey     */
    255,137,229, /* pink     */
    56,203,0,    /* lt green */
    213,213,26,  /* yellow   */
    98,246,153,  /* aqua     */
    255,255,255};/* white    */

/* http://en.wikipedia.org/wiki/List_of_8-bit_computer_hardware_palettes */
uchar wikipedia[16][3] = {
	0,0,0,		 /* black */
	114,38,64,   /* red */
	64,51,127,   /* dk blue */
	228,52,254,  /* purple */
	14,89,64,    /* dk green */
	128,128,128, /* gray */
	27,154,254,  /* med blue */
	191,179,255, /* lt blue */
	64,76,0,     /* brown */
	228,101,1,   /* orange */
	128,128,128, /* grey */
	241,166,191, /* pink */
	27,203,1,    /* lt green */
	191,204,128, /* yellow */
	141,217,191, /* aqua */
	255,255,255};/* white */

/* http://wsxyz.net/tohgr.html */
/* Sheldon Simm's palette from todhr */
/*

Sheldon is clipping the black and white ranges.

The usual reason for doing this is for dirty images (caused by poor digitizing or
sampling or re-sampling color depth loss due to scaling). By using a clipping threshold
at either end of the rgb range, blacks that are not quite black don't match to some other
dark color and whites that are not quite white don't match to some other light color.

I too put this in place as a clipping option when I wrote Bmp2SHR about a year ago in my
Brooks output routines but it worked a little differently. I didn't build it into a
palette for one thing.

Sheldon's weighting favours clipping blue gun values at both ends of the range.

Red is clipped more than green in the high range, and green is clipped more than
red in the low range.

 static Pixel pal[] = {
    {   1,   4,   8}, // 0 black
    {  32,  54, 212}, // 1 dk blue
    {  51, 111,   0}, // 2 dk green
    {   7, 168, 225}, // 3 med blue
    {  99,  77,   0}, // 4 brown
    { 126, 126, 126}, // 5 gray
    {  67, 200,   0}, // 6 lt green
    {  93, 248, 133}, // 7 aqua
    { 148,  12, 125}, // 8 red
    { 188,  55, 255}, // 9 purple
    { 126, 126, 126}, // A grey
    { 158, 172, 255}, // B lt blue
    { 249,  86,  29}, // C orange
    { 255, 129, 236}, // D pink
    { 221, 206,  23}, // E yellow
    { 248, 250, 244}  // F white

    Also note that I use the array name grpal. This was used as a palette in
    one of Sheldon's previous versions. Since I have propagated this array name
    to my code and my code is working fine with it, I have no plans to change it
    to maintain currency with Sheldon's code.

*/
uchar grpal[16][3] = {
	0,0,0,		 /* black */
	148,12,125,  /* red - hgr 0*/
	32,54,212,   /* dk blue - hgr 0 */
	188,55,255,  /* purple - default HGR overlay color */
	51,111,0,    /* dk green - hgr 0 */
	126,126,126, /* gray - hgr 0 */
	7,168,225,   /* med blue */
	158,172,255, /* lt blue - hgr 0 - alternate HGR overlay color*/
	99,77,0,     /* brown - hgr 0 */
	249,86,29,   /* orange */
	126,126,126, /* grey - hgr 0 */
	255,129,236, /* pink - hgr 0 */
	67,200,0,    /* lt green */
	221,206,23,  /* yellow - hgr 0 */
	93,248,133,  /* aqua - hgr 0 */
	255,255,255};/* white */

/* Sheldon still uses the RGB values from his old palette for HGR conversion */
uchar hgrpal[16][3] = {
	0x00,0x00,0x00, /* black */
	0xad,0x18,0x28, /* red */
	0x55,0x1b,0xe1, /* dk blue */
	0xe8,0x2c,0xf8, /* purple 232,44,248 - default hgr overlay color */
	0x01,0x73,0x63, /* dk green */
	0x7e,0x82,0x7f, /* gray */
	0x34,0x85,0xfc, /* med blue - 52,133,252 - alternate HGR overlay color */
	0xd1,0x95,0xff, /* lt blue */
	0x33,0x6f,0x00, /* brown */
	0xd0,0x81,0x01, /* orange */
	0x7f,0x7e,0x77, /* grey */
	0xfe,0x93,0xa3, /* pink */
	0x1d,0xd6,0x09, /* lt green */
	0xae,0xea,0x22, /* yellow */
	0x5b,0xeb,0xd9, /* aqua */
	0xff,0xff,0xff};/* white */

/* imported palette file */
uchar rgbUser[16][3];

uchar SuperConvert[16][3] = {
	0,0,0,		 /* black */
	221,0,51,    /* red */
	0,0,153,     /* dk blue */
	221,0,221,   /* purple */
	0,119,0,     /* dk green */
	85,85,85,    /* gray */
	34,34,255,   /* med blue */
	102,170,255, /* lt blue */
	136,85,34,   /* brown */
	255,102,0,   /* orange */
	170,170,170, /* grey */
	255,153,136, /* pink */
	0,221,0,     /* lt green */
	255,255,0,   /* yellow */
	0,255,153,   /* aqua */
	255,255,255};/* white */

uchar Jace[16][3] = {
	0,0,0,		  /* black */
	177,0,93,	  /* red */
	32,41,255,	  /* dk blue */
	210,41,255,	  /* purple */
	0,127,34,	  /* dk green */
	127,127,127,  /* gray */
	0,168,255,	  /* med blue */
	160,168,255,  /* lt blue */
	94,86,0,	  /* brown */
	255,86,0,	  /* orange */
	127,127,127,  /* grey */
	255,127,220,  /* pink */
	44,213,0,	  /* lt green */
	222,213,0,	  /* yellow */
	77,255,161,	  /* aqua */
	255,255,255}; /* white */

/* https://github.com/cybernesto/VBMP/wiki/Converting-a-picture-into-a-DHGR-color-image-using-the-GIMP */
/* Robert Munafo - http://mrob.com/pub/xapple2/colors.html */
uchar Cybernesto[16][3] = {
	  0,  0,  0,	/* 0  Black       */
	227, 30, 96,	/* 1  Magenta     */
	 96, 78,189,	/* 8  Dark Blue   */
	255, 68,253,	/* 9  Violet  	  */
	  0,163, 96,	/* 4  Dark Green  */
	156,156,156,	/* 5  Grey1    	  */
	 20,207,253,	/* 12 Medium Blue */
	208,195,255,	/* 13 Light Blue  */
	 96,114,  3,	/* 2  Brown       */
	255,106, 60,	/* 3  Orange      */
	156,156,156,	/* 10 Grey2       */
	255,160,208,	/* 11 Pink        */
	 20,245, 60,	/* 6  Green       */
	208,221,141,	/* 7  Yellow      */
	114,255,208,	/* 14 Aqua        */
	255,255,255};	/* 15 White       */

/* merged palette */
/* initial values are the average values of p5 NTSC and p12 RGB */
uchar PseudoPalette[16][3] = {
	0,0,0,
	184,6,88,
	16,27,182,
	204,27,238,
	25,115,0,
	105,105,105,
	20,101,240,
	130,171,255,
	117,81,17,
	252,94,14,
	148,148,148,
	255,141,186,
	33,210,0,
	238,230,11,
	46,251,143,
	255,255,255};

/* goes with the canvas bmp that I put out there with AppleX */
uchar rgbCanvasArray[16][3] = {
0  , 0  , 0  ,
208, 0  , 48 ,
0  , 0  , 128,
255, 0  , 255,
0  , 128, 0  ,
128, 128, 128,
0  , 0  , 255,
96 , 160, 255,
128, 80 , 0  ,
255, 128, 0  ,
192, 192, 192,
255, 144, 128,
0  , 255, 0  ,
255, 255, 0  ,
64 , 255, 144,
255, 255, 255};

/* this might work with old Win16 16 color BMP's */
uchar rgbBmpArray[16][3] = {
0  ,0  , 0  ,
191,0  , 0  ,
0  ,0  , 191,
191,0  , 191,
0  ,191, 0  ,
128,128, 128,
0  ,191, 191,
0  ,0  , 255,
191,191, 0  ,
255,0  , 0  ,
192,192, 192,
255,0  , 255,
0  ,255, 0  ,
255,255, 0  ,
0  ,255, 255,
255,255, 255};

/* this might work with new Win32 16 color BMP's */
uchar rgbXmpArray[16][3] = {
0  , 0  , 0  ,
128, 0  , 0  ,
0  , 0  , 128,
128, 0  , 128,
0  , 128, 0  ,
128, 128, 128,
0  , 128, 128,
0  , 0  , 255,
128, 128, 0  ,
255, 0  , 0  ,
192, 192, 192,
255, 0  , 255,
0  , 255, 0  ,
255, 255, 0  ,
0  , 255, 255,
255, 255, 255};

/* from the bios in some PC I had */
uchar rgbVgaArray[16][3] = {
0  , 0  , 0  ,
255, 0  , 0  ,
0  , 0  , 255,
255, 0  , 255,
0  , 255, 0  ,
85 , 85 , 85 ,
0  , 255, 255,
85 , 85 , 255,
255, 255, 0  ,
255, 85 , 85 ,
192, 192, 192,
255, 85 , 255,
85 , 255, 85 ,
255, 255, 85 ,
85 , 255, 255,
255, 255, 255};

/* some old ZSoft VGA Pcx Colors */
uchar rgbPcxArray[16][3] = {
0  , 0  , 0  ,
170, 0  , 0  ,
0  , 0  , 170,
170, 0  , 170,
0  , 170, 0  ,
85 , 85 , 85 ,
0  , 170, 170,
85 , 85 , 255,
170, 170, 0  ,
255, 85 , 85 ,
170, 170, 170,
255, 85 , 255,
85 , 255, 85 ,
255, 255, 85 ,
85 , 255, 255,
255, 255, 255};

/* our working copy of the apple II double hires colors */
/* this is in Apple II lo-res color order */
/* todhr palette */
uchar rgbArray[NUM_VGA_COLORS][NUM_RGB_COLORS]={
	0,0,0,		 /* black */
	148,12,125,  /* red */
	32,54,212,   /* dk blue */
	188,55,255,  /* purple */
	51,111,0,    /* dk green */
	126,126,126, /* gray */
	7,168,225,   /* med blue */
	158,172,255, /* lt blue */
	99,77,0,     /* brown */
	249,86,29,   /* orange */
	126,126,126, /* grey */
	255,129,236, /* pink */
	67,200,0,    /* lt green */
	221,206,23,  /* yellow */
	93,248,133,  /* aqua */
	255,255,255};/* white */



/* Apple 2 Double Hires Format */

/* provides base address for page1 hires scanlines  */
unsigned HB[]={
0x2000, 0x2400, 0x2800, 0x2C00, 0x3000, 0x3400, 0x3800, 0x3C00,
0x2080, 0x2480, 0x2880, 0x2C80, 0x3080, 0x3480, 0x3880, 0x3C80,
0x2100, 0x2500, 0x2900, 0x2D00, 0x3100, 0x3500, 0x3900, 0x3D00,
0x2180, 0x2580, 0x2980, 0x2D80, 0x3180, 0x3580, 0x3980, 0x3D80,
0x2200, 0x2600, 0x2A00, 0x2E00, 0x3200, 0x3600, 0x3A00, 0x3E00,
0x2280, 0x2680, 0x2A80, 0x2E80, 0x3280, 0x3680, 0x3A80, 0x3E80,
0x2300, 0x2700, 0x2B00, 0x2F00, 0x3300, 0x3700, 0x3B00, 0x3F00,
0x2380, 0x2780, 0x2B80, 0x2F80, 0x3380, 0x3780, 0x3B80, 0x3F80,
0x2028, 0x2428, 0x2828, 0x2C28, 0x3028, 0x3428, 0x3828, 0x3C28,
0x20A8, 0x24A8, 0x28A8, 0x2CA8, 0x30A8, 0x34A8, 0x38A8, 0x3CA8,
0x2128, 0x2528, 0x2928, 0x2D28, 0x3128, 0x3528, 0x3928, 0x3D28,
0x21A8, 0x25A8, 0x29A8, 0x2DA8, 0x31A8, 0x35A8, 0x39A8, 0x3DA8,
0x2228, 0x2628, 0x2A28, 0x2E28, 0x3228, 0x3628, 0x3A28, 0x3E28,
0x22A8, 0x26A8, 0x2AA8, 0x2EA8, 0x32A8, 0x36A8, 0x3AA8, 0x3EA8,
0x2328, 0x2728, 0x2B28, 0x2F28, 0x3328, 0x3728, 0x3B28, 0x3F28,
0x23A8, 0x27A8, 0x2BA8, 0x2FA8, 0x33A8, 0x37A8, 0x3BA8, 0x3FA8,
0x2050, 0x2450, 0x2850, 0x2C50, 0x3050, 0x3450, 0x3850, 0x3C50,
0x20D0, 0x24D0, 0x28D0, 0x2CD0, 0x30D0, 0x34D0, 0x38D0, 0x3CD0,
0x2150, 0x2550, 0x2950, 0x2D50, 0x3150, 0x3550, 0x3950, 0x3D50,
0x21D0, 0x25D0, 0x29D0, 0x2DD0, 0x31D0, 0x35D0, 0x39D0, 0x3DD0,
0x2250, 0x2650, 0x2A50, 0x2E50, 0x3250, 0x3650, 0x3A50, 0x3E50,
0x22D0, 0x26D0, 0x2AD0, 0x2ED0, 0x32D0, 0x36D0, 0x3AD0, 0x3ED0,
0x2350, 0x2750, 0x2B50, 0x2F50, 0x3350, 0x3750, 0x3B50, 0x3F50,
0x23D0, 0x27D0, 0x2BD0, 0x2FD0, 0x33D0, 0x37D0, 0x3BD0, 0x3FD0};


unsigned char dhrbuf[16384];

/*

The following is logically reordered to match the lores
color order...

                                                Repeated
                                                Binary
          Color         aux1  main1 aux2  main2 Pattern
          Black          00    00    00    00    0000
          Magenta        08    11    22    44    0001
		  Dark Blue      11    22    44    08    1000
          Violet         19    33    66    4C    1001
          Dark Green     22    44    08    11    0100
          Grey1          2A    55    2A    55    0101
          Medium Blue    33    66    4C    19    1100
          Light Blue     3B    77    6E    5D    1101
          Brown          44    08    11    22    0010
          Orange         4C    19    33    66    0011
          Grey2          55    2A    55    2A    1010
          Pink           5D    3B    77    6E    1011
          Green          66    4C    19    33    0110
          Yellow         6E    5D    3B    77    0111
          Aqua           77    6E    5D    3B    1110
          White          7F    7F    7F    7F    1111
*/

/*

#define LOBLACK   	0
#define LORED     	1
#define LODKBLUE 	2
#define LOPURPLE  	3
#define LODKGREEN	4
#define LOGRAY    	5
#define LOMEDBLUE	6
#define LOLTBLUE 	7
#define LOBROWN   	8
#define LOORANGE  	9
#define LOGREY    	10
#define LOPINK    	11
#define LOLTGREEN	12
#define LOYELLOW  	13
#define LOAQUA    	14
#define LOWHITE   	15

*/

/* the following array is based on the above */
unsigned char dhrbytes[16][4] = {
	0x00,0x00,0x00,0x00,
	0x08,0x11,0x22,0x44,
	0x11,0x22,0x44,0x08,
	0x19,0x33,0x66,0x4C,
	0x22,0x44,0x08,0x11,
	0x2A,0x55,0x2A,0x55,
	0x33,0x66,0x4C,0x19,
	0x3B,0x77,0x6E,0x5D,
	0x44,0x08,0x11,0x22,
	0x4C,0x19,0x33,0x66,
	0x55,0x2A,0x55,0x2A,
	0x5D,0x3B,0x77,0x6E,
	0x66,0x4C,0x19,0x33,
	0x6E,0x5D,0x3B,0x77,
	0x77,0x6E,0x5D,0x3B,
	0x7F,0x7F,0x7F,0x7F};

/* DHGR file input Helper Function for AUX, BIN file pairs */
/* a double hi-res color pixel can occur at any one of 7 positions */
/* in a 4 byte block which spans aux and main screen memory */
/* the horizontal resolution is 140 pixels */
/* read 2 input files */
int read_binaux(uchar *basename)
{

	FILE *fp;
	uchar infile[256];

    /* the bsaved images are split into two files
       the first file is loaded into aux mem */
    sprintf(infile,"%s.AUX",basename);
	fp = fopen(infile,"rb");

	if (NULL == fp && longnames == 1) {
		sprintf(infile,"%s.AUX#062000",basename);
        fp = fopen(infile,"rb");
	}
	if (NULL == fp)return INVALID;
	fread(dhrbuf,1,8192,fp);
	fclose(fp);

    /* the second file is loaded into main mem */
    sprintf(infile,"%s.BIN",basename);
	fp = fopen(infile,"rb");

	if (NULL == fp && longnames == 1) {
		sprintf(infile,"%s.BIN#062000",basename);
        fp = fopen(infile,"rb");
	}
	if (NULL == fp)return INVALID;
	fread(&dhrbuf[8192],1,8192,fp);
	fclose(fp);

	return SUCCESS;

}

/* DHGR file input Helper Function for A2FC files */
/* read one input file */
int read_2fc(uchar *basename)
{
	FILE *fp;
	uchar infile[256];

	if (longnames == 0) {
		sprintf(infile,"%s.2FC",basename);
		fp = fopen(infile,"rb");
	}
	else {
    	for (;;) {
			/* support for longnames and ciderpress tags */
			/* support for tohgr output */
			if (tohgr == 0)
				sprintf(infile,"%s.2FC",basename);
			else
				sprintf(infile,"%s.dhgr",basename);

            fp = fopen(infile,"rb");
            if (NULL != fp)break;

            if (tohgr == 0)
            	sprintf(infile,"%s.A2FC",basename);
			else
				sprintf(infile,"%s.DHGR",basename);

            fp = fopen(infile,"rb");
			if (NULL != fp)break;

			if (tohgr == 0)
				sprintf(infile,"%s.2FC#062000",basename);
			else
				sprintf(infile,"%s.dhgr#062000",basename);

            fp = fopen(infile,"rb");
			if (NULL != fp)break;

            if (tohgr == 0)
            	sprintf(infile,"%s.A2FC#062000",basename);
			else
				sprintf(infile,"%s.DHGR#062000",basename);

            fp = fopen(infile,"rb");
            break;
		}
	}

 	if (NULL == fp)return INVALID;
	fread(dhrbuf,1,16384,fp);
	fclose(fp);

	return SUCCESS;

}

/* Color DHGR Pixel-Mapping  Helper Function */
/* returns the Apple II Double Hi-res drawcolor 0-15 */
/* a double hi-res color pixel can occur at any one of 7 positions */
/* in a 4 byte block which spans aux and main screen memory */
/* the horizontal resolution is 140 pixels */
int dhrgetpixel(int x,int y)
{
    int xoff, pattern, idx;
    unsigned char *ptraux, *ptrmain,c1, c2, d1, d2;

    pattern = (x%7);
	xoff = HB[y] + ((x/7) * 2);
    ptraux  = (unsigned char *) &dhrbuf[xoff-0x2000];
    ptrmain = (unsigned char *) &dhrbuf[xoff];


	switch(pattern)
	{
		/* left this here for reference

		unsigned char dhrpattern[7][4] = {
		0,0,0,0,
		0,0,0,1,
		1,1,1,1,
		1,1,2,2,
		2,2,2,2,
		2,3,3,3,
        3,3,3,3};
        */

        /* compare colors in the input file to color patterns and return drawcolor */
        /* somewhat inelegant but lazy to read and debug if a problem */
		case 0: c1 = ptraux[0] &0x0f;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][0] & 0x0f;
				  if (d1 == c1) return idx;
				}
		        break;
		case 1: c1 = ptraux[0] & 0x70;
		        c2 = ptrmain[0] & 0x01;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][0] & 0x70;
				  d2 = dhrbytes[idx][1] & 0x01;
				  if (d1 == c1 && d2 == c2) return idx;
				}
		        break;
		case 2: c1 = ptrmain[0] & 0x1e;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][1] & 0x1e;
				  if (d1 == c1) return idx;
				}
		        break;
		case 3: c1 = ptrmain[0] & 0x60;
		        c2 = ptraux[1] & 0x03;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][1] & 0x60;
				  d2 = dhrbytes[idx][2] & 0x03;
				  if (d1 == c1 && d2 == c2) return idx;
				}
                break;
		case 4: c1 = ptraux[1] & 0x3c;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][2] & 0x3c;
				  if (d1 == c1) return idx;
				}
		        break;
		case 5: c1 = ptraux[1] & 0x40;
 		        c2 = ptrmain[1] & 0x07;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][2] & 0x40;
				  d2 = dhrbytes[idx][3] & 0x07;
				  if (d1 == c1 && d2 == c2) return idx;
				}
                break;
		case 6: c1 = ptrmain[1] & 0x78;
		        for (idx = 0; idx < 16; idx++) {
				  d1 = dhrbytes[idx][3] & 0x78;
				  if (d1 == c1) return idx;
				}
 		        break;
	}

    return INVALID;
}

/* Color Output Helper Function */
int save_to_bmp24(uchar *basename)
{

    FILE *fp;
    uchar outfile[256], tempr, tempg, tempb;
	int x,y,y2,idx;


    sprintf(outfile,"%s.bmp",basename);
	fp = fopen(outfile,"wb");
	if (NULL == fp)return INVALID;

	/* write rgb triples and double each pixel to preserve the aspect ratio */
	if (doublepixel == 1)
	    /* write header for 280 x 192 x 24 bit bmp */
    	fwrite(BMP_header,1,sizeof(BMP_header),fp);
    else
        /* write header for 140 x 192 x 24 bit bmp */
        fwrite(BMP140_header,1,sizeof(BMP140_header),fp);

    y2 = 191;
   	for (y = 0; y< 192; y++) {

	   for (x = 0; x < 140; x++) {
		  idx = dhrgetpixel(x,y2);

          /* range check */
          if (idx < 0 || idx > 15)idx = 0; /* default to black */

		  tempr = rgbArray[idx][0];
		  tempg = rgbArray[idx][1];
		  tempb = rgbArray[idx][2];

		  /* reverse order */
		  fputc(tempb, fp);
		  fputc(tempg, fp);
		  fputc(tempr, fp);

		  if (doublepixel == 1) {
		  	 /* double-up */
		  	 fputc(tempb, fp);
		  	 fputc(tempg, fp);
		  	 fputc(tempr, fp);
		  }
	   }
	   y2 -= 1;
    }

    fclose(fp);
    return SUCCESS;

}

unsigned char AppleWinLine[560];

/* encodes apple II dhgr scanline into buffer */
int applewinbits(int y)
{
	    int xoff,idx,jdx;
	    unsigned char *ptraux, *ptrmain, bits[7];

	    xoff = HB[y];
	    ptraux  = (unsigned char *) &dhrbuf[xoff-0x2000];
	    ptrmain = (unsigned char *) &dhrbuf[xoff];

        xoff = 0;
        for (idx = 0; idx < 40; idx++) {

            for (jdx = 0; jdx < 7; jdx++) {
				bits[jdx] = AppleWinLine[xoff]; xoff++;
			}
			ptraux[idx] = (bits[6]<<6|bits[5]<<5|bits[4]<<4|
                           bits[3]<<3|bits[2]<<2|bits[1]<<1|bits[0]);

            for (jdx = 0; jdx < 7; jdx++) {
				bits[jdx] = AppleWinLine[xoff]; xoff++;
			}
			ptrmain[idx] = (bits[6]<<6|bits[5]<<5|bits[4]<<4|
                            bits[3]<<3|bits[2]<<2|bits[1]<<1|bits[0]);

		}
		return SUCCESS;

}

/* encodes apple II dhgr scanline into buffer */
int applebits(int y)
{
	    int xoff,idx,jdx;
	    unsigned char *ptraux, *ptrmain, bits[7];

	    xoff = HB[y];
	    ptraux  = (unsigned char *) &dhrbuf[xoff-0x2000];
	    ptrmain = (unsigned char *) &dhrbuf[xoff];

        xoff = 0;
        for (idx = 0; idx < 40; idx++) {

            for (jdx = 0; jdx < 7; jdx++) {
				bits[jdx] = buf560[xoff]; xoff++;
			}
			ptraux[idx] = (bits[6]<<6|bits[5]<<5|bits[4]<<4|
                           bits[3]<<3|bits[2]<<2|bits[1]<<1|bits[0]);

            for (jdx = 0; jdx < 7; jdx++) {
				bits[jdx] = buf560[xoff]; xoff++;
			}
			ptrmain[idx] = (bits[6]<<6|bits[5]<<5|bits[4]<<4|
                            bits[3]<<3|bits[2]<<2|bits[1]<<1|bits[0]);

		}
		return SUCCESS;

}

/* Input File Helper Function - called from main() */
/* Reads an AppleWin Monochrome Screen Capture (256 color BMP) */
/* Writes an Apple II Compatible DHGR A2FC file (or optionally AUX, BIN file pair) */
/* Renames AppleWin Screen Capture from BMP to BM2 */
/* The Apple II Compatible Output is then processed normally. */
int ReadAppleWin(unsigned char *basename, unsigned char *newname)
{

 	FILE *fp;
    int packet = INVALID, y,i,j;
    char bmpfile[256], outfile[256];

	if (bm2 == 1) sprintf(bmpfile,"%s.bm2",basename);
	else sprintf(bmpfile,"%s.bmp",basename);

    if((fp=fopen(bmpfile,"rb"))==NULL) {
		if (bm2 == 1) return INVALID;
		sprintf(bmpfile,"%s.bm2",basename);
		if((fp=fopen(bmpfile,"rb"))==NULL) {
			return INVALID;
		}
		bm2 = 1;
	}

    /* read the header stuff into the appropriate structures,
       it's likely a bmp file */
    memset(&BitMapFileHeader.bfType,0,sizeof(BITMAPFILEHEADER));
    memset(&bmi.biSize,0,sizeof(BITMAPINFOHEADER));

    fread((char *)&BitMapFileHeader.bfType,
	             sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmi.biSize,
                 sizeof(BITMAPINFOHEADER),1,fp);

    /* check for an applewin bmp */
    if (bmi.biCompression==BI_RGB &&
        BitMapFileHeader.bfType[0] == 'B' && BitMapFileHeader.bfType[1] == 'M' &&
        bmi.biPlanes==1 && bmi.biBitCount == 8 &&
        bmi.biWidth == 560 &&  bmi.biHeight == 384)  {
		fread((char *)&sbmp[0].rgbBlue, sizeof(RGBQUAD)*256,1,fp);
		if (sbmp[17].rgbRed == 255 && sbmp[17].rgbGreen == 255 && sbmp[17].rgbBlue == 254) packet = 560;
	}

    if(packet == INVALID){
		puts(szTextTitle);
		printf("%s is not a Supported 560 x 384 AppleWin Screen Capture.\n", bmpfile);
		fclose(fp);
		exit(1);
	}


	memset(&dhrbuf[0],0,16384); /* clear write buffer */


    i = 191; /* bmp's are upside-down
                so conversion of scanlines is in
                reverse order */
    for(y=192;y>0;y--)
    {
	  fread((char *)&AppleWinLine[0],1,560,fp);
	  /* skip every second scanline  */
	  fread((char *)&AppleWinLine[0],1,560,fp);
	  for (j=0;j<560;j++) {
	  	if (AppleWinLine[j] != 0)AppleWinLine[j] = 1;
	  }
	  if (mono == 1) applebits(i);
	  else applewinbits(i); /* encode 7 bit mono line */
      i--;
      if (i< 0) break;
    }
    fclose(fp);

    /* keep output files to a minimum */
    /* output Apple II files in one format only */
	if (applesoft == 0) {
    	if (longnames == 0)sprintf(outfile,"%s.2FC", newname);
    	else sprintf(outfile,"%s.A2FC", newname);
		fp = fopen(outfile,"wb");
		if (NULL == fp) {
			puts(szTextTitle);
			printf("%s cannot be created.\n", outfile);
			exit(1);
		}
		fwrite(dhrbuf,1,16384,fp);
		fclose(fp);
	}
	else {
		/* output AUX,BIN - default is A2FC */
    	/* the bsaved images are split into two files
       	the first file is loaded into aux mem */
    	sprintf(outfile,"%s.AUX",newname);
		fp = fopen(outfile,"wb");
		if (NULL == fp) {
			puts(szTextTitle);
			printf("%s cannot be created.\n", outfile);
			exit(1);
		}
		fwrite(dhrbuf,1,8192,fp);
		fclose(fp);

    	/* the second file is loaded into main mem */
    	sprintf(outfile,"%s.BIN",newname);
		fp = fopen(outfile,"wb");
		if (NULL == fp) {
			puts(szTextTitle);
			printf("%s cannot be created.\n", outfile);
			exit(1);
		}
		fwrite(&dhrbuf[8192],1,8192,fp);
		fclose(fp);
	}

    /* save a back-up of the original AppleWin file */
    /* if we are not already processing the back-up */
    if (bm2 == 0) {
		sprintf(outfile,"%s.bm2",basename);
		remove(outfile);
		rename(bmpfile,outfile);
	}

return SUCCESS;

}


/* External Palette File - Color Output Helper functions */

/* These were put in place primarily to support:

1. The GIMP's gpl palette file format.
2. A simple CSV (comma separated value, comma delimited) file

Support for other types of palette files is speculative. Nobody might ever use
the others but the routines were already written so no harm to include the others

External palette files must be in the Lo-Res Color Order.

*/

/* ------------------------------------------------------------------------ */
/* START of palette reader and helper functions adapted from Clipshop       */
/* ------------------------------------------------------------------------ */

/* strip line feeds from ascii file lines... */

void nocr(char *ptr) {
  int idx;
  for (idx = 0; ptr[idx] != 0; idx++)
    if (ptr[idx] == LFEED || ptr[idx] == CRETURN || ptr[idx] == '#')
      ptr[idx] = 0;
}

/*
squeeze redundant whitespace from lines read-in from a palette file
(leave only a single space character)
this is important if the user has created their own palette file
by hand... since they may accidentally type more than one whitespace
between RGB values...

Also, phototsyler version 2 palette file lines are fixed width,
right justified so we need to massage these for our reader...
*/
void SqueezeLine(char *ptr)
{
  int idx, jdx, len;
  char buf[128];

  idx = 0;
  while (ptr[idx] == ' ')idx++;  /* remove leading whitespace */
  strcpy(buf, &ptr[idx]);

  jdx = 0;
  ptr[jdx] = ASCIIZ;

  for (idx = 0; buf[idx] != ASCIIZ; idx++) {
    if (buf[idx] == 9) buf[idx] = ' ';         /* no tabs please */
    if (buf[idx] == ',') buf[idx] = ' ';       /* no commas please */
    if (buf[idx] == ' ' && buf[idx +1] == ' ')
      continue;
    /* truncate if any non-numeric characters */
    if ((buf[idx] < '0' || buf[idx] > '9') && buf[idx] != ' ')
      buf[idx] = ASCIIZ;
    ptr[jdx] = buf[idx]; jdx++;
    ptr[jdx] = ASCIIZ;
  }

  /* remove trailing whitespace...
    this occurrs during parsing of photostyler */
  len = strlen(ptr);
  while (len > 0) {
    len--;
    if (ptr[len] != ' ')
      break;
    ptr[len] = ASCIIZ;
  }
}

/* split the RGB triple from a text line read-in from an
   ascii palette file. */
int ReadPaletteLine(unsigned char *ptr, unsigned char *palptr, unsigned int colordepth)
{
  int red, green, blue, idx, spaces = 0;

  red = atoi(ptr);
  if (red < 0 || red > 255) return INVALID;

  /* there must be at least 3 fields */
  for (idx = 0; ptr[idx] != 0; idx++) {
    if (ptr[idx] == ' ' && ptr[idx+1] >= '0' && ptr[idx+1] <= '9') {
       spaces++;
       switch(spaces) {
         case 1:
           green = atoi(&ptr[idx+1]);
           if (green < 0 || green > 255) return INVALID;
           break;
         case 2:
           blue = atoi(&ptr[idx+1]);
           if (blue < 0 || blue > 255) return INVALID;
           break;
       }
    }
  }

  if (spaces<2)
    return INVALID;

  if (colordepth == 6) {
     palptr[0] = (uchar)red << 2;
     palptr[1] = (uchar)green << 2;
     palptr[2] = (uchar)blue << 2;
   }
   else {
     palptr[0] = (uchar)red;
     palptr[1] = (uchar)green;
     palptr[2] = (uchar)blue;
   }
   return SUCCESS;

}

/* check version if Paintshop palette since JASC may change someday */
/* also check Aldus version although that product is old... */

/* The Top Half of NeoPaint Windows Palettes are the same as their */
/* DOS palettes so we use the 6 bit color values and handle both   */
/* file types the same way... so no worry about neopaint versions. */

char *Gimp = "GIMP Palette"; /* followed by RGB values and comments */

/* NeoPaint and PaintShop Pro headers
   3 lines followed by RGB values */
char *NeoPaint  = "NeoPaint Palette File";
char *PaintShop = "JASC-PAL";
char *PaintShopVersion = "0100";

/* Aldus photostyler
   3 lines followed by RGB values */
char *AldusPal = "CWPAL";
char *AldusClr = "CWCLR";           /* partial palettes */
char *AldusVersion = "100";

#define GENERIC 1
#define GIMP 2
#define JASC 3
#define NEO 4
#define ALDUS 5

sshort GetUserPalette(char *name)
{
	FILE *fp;
	char buf[128];
	int cnt=16;
	sshort status = INVALID;
	unsigned colordepth=8,paltype=GENERIC;

	fp = fopen(name,"r");
	if (fp == NULL) return status;

	for (;;) {
  		if (NULL == fgets(buf, 128, fp)) {
    		fclose(fp);
    		break;
		}
  		nocr(buf);
  		SqueezeLine(buf);

        /* check for some known palette types */
  		if (strcmp(Gimp, buf)==0) paltype = GIMP;
  		else if (strcmp(PaintShop, buf)==0) paltype = JASC;
  		else if (strcmp(NeoPaint, buf)==0) {
  			colordepth = 6;
    		paltype = NEO;
  		}
  		else if (strcmp(AldusPal, buf) == 0 || strcmp(AldusClr, buf) == 0) {
	  		paltype = ALDUS;
  		}
  		/* if not a known type then assume it's just a simple csv */

  		status = SUCCESS;
		switch(paltype)
		{
			case GENERIC: rewind(fp); break;

			case JASC:
			case NEO:
			case ALDUS:
			    /* check 2 remaining header lines */
				status = INVALID;
				if (NULL == fgets(buf, 128, fp)) break;
				nocr(buf);
				SqueezeLine(buf);
				if (paltype == JASC && strcmp(PaintShopVersion, buf)!=0)break;
				if (paltype == ALDUS && strcmp(AldusVersion, buf) != 0)break;
				if (NULL == fgets(buf, 128, fp)) break;
				cnt = atoi(buf);
				if (cnt < 16) break;
				status = SUCCESS;
		}
		if (status == INVALID) break;

        memset(&rgbUser[0][0],0,48);
        cnt = 0;
		while (fgets(buf,128,fp) != NULL) {
			if (buf[0] == '#') continue;
			if (strlen(buf) < 5) continue;
  			nocr(buf);
  			SqueezeLine(buf);
    		if (INVALID == ReadPaletteLine(buf,(uchar *)&rgbUser[cnt][0],colordepth)) continue;
    		cnt++;
    		if (cnt > 15)break;
		}
		break;
	}
	fclose(fp);

	if (cnt < 15) {
		printf("%s contains only %d colors!",name,cnt);
	}
	if (status == INVALID) {
		printf("%s is not a valid palette file!",name);
	}
	return status;
}

/* ------------------------------------------------------------------------ */
/* END of palette reader and helper functions adapted from Clipshop         */
/* ------------------------------------------------------------------------ */

/* Color Output Helper Function */
/* select conversion palette based on command-line options */
/* matches Bmp2DHR */
int GetBuiltinPalette(uchar cmd, uchar value, int palidx)
{
	int i,j, singlegrey = 0;
	uchar r,g,b;
	unsigned temp;

    if (cmd != 'P') return -1;
    if (value < 48 || value > 57) return -1;


	switch(palidx) {
		case 16:/* HGR conversion - optional palette from tohgr */
		        for (i=0;i<16;i++) {
					rgbArray[i][0] = hgrpal[i][0];
					rgbArray[i][1] = hgrpal[i][1];
					rgbArray[i][2] = hgrpal[i][2];
				}
				break;
		case 15: for (i=0;i<16;i++) {
					rgbArray[i][0] = PseudoPalette[i][0];
					rgbArray[i][1] = PseudoPalette[i][1];
					rgbArray[i][2] = PseudoPalette[i][2];
				}
				break;
		/* Cybernesto */
		case 14: for (i=0;i<16;i++) {
					rgbArray[i][0] = Cybernesto[i][0];
					rgbArray[i][1] = Cybernesto[i][1];
					rgbArray[i][2] = Cybernesto[i][2];
				}
				singlegrey = 1;
				break;
		/* Jace */
		case 13: for (i=0;i<16;i++) {
					rgbArray[i][0] = Jace[i][0];
					rgbArray[i][1] = Jace[i][1];
					rgbArray[i][2] = Jace[i][2];
				}
				singlegrey = 1;
				break;
		/* Super Convert */
		case 12: for (i=0;i<16;i++) {
					rgbArray[i][0] = SuperConvert[i][0];
					rgbArray[i][1] = SuperConvert[i][1];
					rgbArray[i][2] = SuperConvert[i][2];
				}
				break;
		/* 5 legacy palettes from BMPA2FC */
		case 11: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbPcxArray[i][0];
					rgbArray[i][1] = rgbPcxArray[i][1];
					rgbArray[i][2] = rgbPcxArray[i][2];
				}
				break;
		case 10: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbVgaArray[i][0];
					rgbArray[i][1] = rgbVgaArray[i][1];
					rgbArray[i][2] = rgbVgaArray[i][2];
				}
				break;
		case 9: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbXmpArray[i][0];
					rgbArray[i][1] = rgbXmpArray[i][1];
					rgbArray[i][2] = rgbXmpArray[i][2];
				}
				break;
		case 8: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbBmpArray[i][0];
					rgbArray[i][1] = rgbBmpArray[i][1];
					rgbArray[i][2] = rgbBmpArray[i][2];
				}
				break;
		case 7: for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbCanvasArray[i][0];
					rgbArray[i][1] = rgbCanvasArray[i][1];
					rgbArray[i][2] = rgbCanvasArray[i][2];
				}
				break;

        /* user definable palette file */
		case 6:
				for (i=0;i<16;i++) {
					rgbArray[i][0] = rgbUser[i][0];
					rgbArray[i][1] = rgbUser[i][1];
					rgbArray[i][2] = rgbUser[i][2];
				}
				break;
        /* tohgr*/
		case 5: for (i=0;i<16;i++) {
					rgbArray[i][0] = grpal[i][0];
					rgbArray[i][1] = grpal[i][1];
					rgbArray[i][2] = grpal[i][2];
				}
				singlegrey = 1;
				break;
        /* wikipedia */
		case 4: for (i=0;i<16;i++) {
					rgbArray[i][0] = wikipedia[i][0];
					rgbArray[i][1] = wikipedia[i][1];
					rgbArray[i][2] = wikipedia[i][2];
				}
				singlegrey = 1;
				break;

		case 3: for (i=0;i<16;i++) {
					rgbArray[i][0] = awinnewcolors[i][0];
					rgbArray[i][1] = awinnewcolors[i][1];
					rgbArray[i][2] = awinnewcolors[i][2];
				}
				break;
		case 2: for (i=0;i<16;i++) {
					rgbArray[i][0] = awinoldcolors[i][0];
					rgbArray[i][1] = awinoldcolors[i][1];
					rgbArray[i][2] = awinoldcolors[i][2];
				}
				break;
		case 1: for (i=0;i<16;i++) {
					rgbArray[i][0] = ciderpresscolors[i][0];
					rgbArray[i][1] = ciderpresscolors[i][1];
					rgbArray[i][2] = ciderpresscolors[i][2];
				}
				break;
		case 0: for (i=0;i<16;i++) {
					rgbArray[i][0] = kegs32colors[i][0];
					rgbArray[i][1] = kegs32colors[i][1];
					rgbArray[i][2] = kegs32colors[i][2];
				}
				break;
		default: return -1;
		}

        /* grey conversion may be necessary if the Apple II Image
           uses light grey and dark grey. */
		if (doublegrey == 1) {
			if (singlegrey != 0 && rgbArray[5][0] != 0) {
			   /* grey 1 = 5, grey2 = 10 */
			   /* use the median between the existing grey and white */
			   /* to approximate the color distance for this palette */
			   temp = 255;
			   temp += rgbArray[5][0];
			   rgbArray[10][0] = rgbArray[10][1] = rgbArray[10][2] = (uchar) (temp/2);
			}
		}



		return palidx;

}


/* Monochrome Output Helper Function */
/* decodes apple II dhgr buffer into 8 bit monochrome scanline buffer */
int applebites(int y)
{
	    int xoff,idx;
	    unsigned char *ptraux, *ptrmain, ch;

	    xoff = HB[y];
	    ptraux  = (unsigned char *) &dhrbuf[xoff-0x2000];
	    ptrmain = (unsigned char *) &dhrbuf[xoff];

        xoff = 0;
        for (idx = 0; idx < 40; idx++) {

            ch = ptraux[idx];

            buf560[xoff] = ((ch) &1); xoff++;
            buf560[xoff] = ((ch >> 1) &1); xoff++;
            buf560[xoff] = ((ch >> 2) &1); xoff++;
            buf560[xoff] = ((ch >> 3) &1); xoff++;
            buf560[xoff] = ((ch >> 4) &1); xoff++;
            buf560[xoff] = ((ch >> 5) &1); xoff++;
            buf560[xoff] = ((ch >> 6) &1); xoff++;

            ch = ptrmain[idx];

            buf560[xoff] = ((ch) &1); xoff++;
            buf560[xoff] = ((ch >> 1) &1); xoff++;
            buf560[xoff] = ((ch >> 2) &1); xoff++;
            buf560[xoff] = ((ch >> 3) &1); xoff++;
            buf560[xoff] = ((ch >> 4) &1); xoff++;
            buf560[xoff] = ((ch >> 5) &1); xoff++;
            buf560[xoff] = ((ch >> 6) &1); xoff++;

 		}
		return SUCCESS;

}

/* Monochrome Output Helper Function */
/* encodes monochrome bmp scanline from 8 bit monochrome scanline buffer */
int ibmbites()
{
     int i,j,k;
     unsigned char bits[8];

     j=0;
     for(i=0;i<70;i++)
     {
        for(k=0;k<8;k++)
        {
		  bits[k] = buf560[j]; j++;
		}
		bmpscanline[i] = (bits[0]<<7|bits[1]<<6|bits[2]<<5|bits[3]<<4|
                          bits[4]<<3|bits[5]<<2|bits[6]<<1|bits[7]);
     }
     return SUCCESS;

}

/* Monochrome Output Helper Function */
/* save a monochrome bmp file in either supported resolution */
int save_to_bmp(unsigned char *basename, int doublepixel)
{

    FILE *fp;
    unsigned char outfile[256];
	int y,y2;


    sprintf(outfile,"%s.bmp",basename);
	fp = fopen(outfile,"wb");
	if (NULL == fp)return INVALID;

    /* write header */
    if (doublepixel == 1)
    	fwrite(mono384,1,sizeof(mono384),fp);
    else
        fwrite(mono192,1,sizeof(mono192),fp);

    /* write scanlines */
    y2 = 191;
    bmpscanline[70] = bmpscanline[71] = 0xff; /* white padding */
   	for (y = 0; y< 192; y++) {
       applebites(y2);
       ibmbites();
       fwrite(bmpscanline,1,72,fp);
       /* double-up */
	   if (doublepixel == 1)
    		fwrite(bmpscanline,1,72,fp);
	   y2 -= 1;
    }

    return SUCCESS;

}


/* Command Line Options Helper Function */
int cmpstr(char *str, char *cmp)
{
	int i;

	if (strlen(cmp) != strlen(str)) return INVALID;
	for (i=0;str[i] != 0;i++) {
		if (toupper(cmp[i]) != toupper(str[i])) return INVALID;
	}
	return SUCCESS;
}

int main(int argc, char **argv)
{

  int status = 0, idx, jdx;
  char fname[256],sname[256],outfile[256], c, d, e, f;
  uchar *wordptr;
  FILE *fp;

  fname[0] = sname[0] = outfile[0] = ASCIIZ;

#ifdef MSDOS
  longnames = 0;
  system("cls");
#endif

  if(argc == 1) {
    puts(szTextTitle);
    puts("Command line Usage:     a2b filenames options");
    puts("                       \"a2b MyAppleWinDHiresMonoScreenCapture.bmp\"");
    puts("                   or  \"a2b MyDHires.2FC\" or \"a2b MyDHires.A2FC\"");
    puts("                   or  \"a2b MyDHires.dhgr\"");
    puts("                   or  \"a2b MyDHires.BIN\" or \"a2b MyDHires.AUX\"");
    puts("If converting .BIN and .AUX file pairs, both must be present.");
    puts("Default is automatic naming. Optional different output filename.");
    puts("                       \"a2b MyDHires.2FC OutfileBaseName\"");
    puts("Option P - Palettes p0-p16");
    puts("Output: 280 x 192 x 24 Bit Windows .BMP File - Default");
    puts("        140 x 192 x 24 Bit Windows .BMP File - Option 140");
    puts("        560 x 384 x Monochrome Windows .BMP File - Option 384");
    puts("        560 x 192 x Monochrome Windows .BMP File - Option 192");


    printf("Enter Input FileName (Blank to Exit): ");
    gets(fname);
    if (fname[0] == ASCIIZ)
      return 1;
    printf("Enter Output FileBaseName (Blank for None) : ");
    gets(outfile);
  }
  else {
    strcpy(fname, argv[1]);
    /* getopts */
    if (argc > 2) {
    	for (idx = 2; idx < argc; idx++) {
			/* switch character is optional */
	   		wordptr = (uchar *)&argv[idx][0];
	   		c = toupper(wordptr[0]);
	   		if (c == '-') {
				wordptr = (uchar *)&argv[idx][1];
                c = toupper(wordptr[0]);
			}
            d = toupper(wordptr[1]);

            if ((c == 'A' && d == (char)ASCIIZ) || cmpstr(wordptr,"BIN") == SUCCESS) {
				/* output AUX,BIN - default is A2FC */
				applesoft = 1;
				continue;
			}
			jdx = atoi((char *)&wordptr[0]);
			if (jdx == 140) {
				/* 140 x 192 x 24-bit color BMP output option - default is 280 x 192 24-bit color BMP */
				doublepixel = 0;
				continue;
			}
			if (c == 'M' || jdx == 384 || jdx == 192 || cmpstr(wordptr,"mono") == SUCCESS) {
				/* 560 x 384 or 560 x 192 monochrome BMP output option - default is 280 x 192 24-bit color BMP */
				mono = 1;
				if (jdx == 192) {
					doublepixel = 0;
				}
				continue;
			}
			if (c == 'P') {
                /* always check for a user definable palette file */
				jdx = GetUserPalette((char *)&wordptr[1]);
				if (jdx == SUCCESS) {
					jdx = 6;
					d = '6';
				}
				else {
					jdx = atoi((char *)&wordptr[1]);
				}
		    	/* palette option - p0 to p16 */
		    	jdx = GetBuiltinPalette(c,d,jdx);
		    	if (jdx != -1) {
					palnumber = jdx;
					continue;
				}
			}
			if (cmpstr(wordptr,"full") == SUCCESS){
				doublegrey = 1;
				continue;
			}
			/* this is somewhat problematic */
			/* however some users may find it convenient to output to a different basename */
			/* I primarily left this in place to match the older MS-DOS version of this utility
			   because a fair amount of documentation is already in place for that. */
		    strcpy(outfile, argv[idx]);
		}
  	}
  }


	if (mono == 1)
		puts("Palette : Black and White Monochrome");
	else
		printf("Palette : %d - %s\n",palnumber,palname[palnumber]);

  jdx = 999;
  for (idx = 0; fname[idx] != ASCIIZ; idx++) {
      if (fname[idx] == '.') jdx = idx;
  }

  if (jdx != 999)
  {
	  strcpy(sname, fname);
	  sname[jdx] = ASCIIZ;
	  if (outfile[0] == ASCIIZ)strcpy(outfile,sname);

	  c = toupper(fname[jdx + 1]);
	  d = toupper(fname[jdx + 2]);
	  e = toupper(fname[jdx + 3]);

      if (longnames != 0) {
	  	f = toupper(fname[jdx + 4]);
      	if (c == 'A' && d == '2' && e == 'F' && f == 'C') a2fc = 1;
	  	/* support for Sheldon Simms tohgr */
	  	if (c == 'D' && d == 'H' && e == 'G' && f == 'R') a2fc = tohgr = 1;
	  }

	  /* support for our extensions only */
	  if (c == 'B' && d == 'M' && e == 'P') bmp = 1;
	  /* process back-up if requested */
	  if (c == 'B' && d == 'M' && e == '2') bmp = bm2 = 1;
	  if (c == '2' && d == 'F' && e == 'C') a2fc = 1;
	  if (c == 'B' && d == 'I' && e == 'N') auxbin = 1;
	  if (c == 'A' && d == 'U' && e == 'X') auxbin = 1;
   }

  status = INVALID;

  if (bmp == 1) {
	  status = ReadAppleWin(sname,outfile);
      if (status == SUCCESS) {
		  if (applesoft == 1) status = read_2fc(outfile);
		  else status = read_binaux(outfile);
	  }
  }
  if (auxbin == 1) status = read_binaux(sname);
  if (a2fc == 1) status = read_2fc(sname);

  if (status) {
    puts(szTextTitle);
    printf("%s is an Unsupported Format or cannot be opened.\n", fname);
    return 1;
  }

  if (mono == 1) status = save_to_bmp(outfile, doublepixel);
  else status = save_to_bmp24(outfile);

 	if (status == SUCCESS) {
	 	printf("%s.BMP Saved!\n",outfile);
 	}
 	else {
	 	printf("Error saving %s.BMP!\n",outfile);
	 	status = 1;
 	}
#ifdef MSDOS
	puts("Have a Nice Dos!");
#endif

 	return status;
}

